-- BigHealthDisplay v5.0: Complete health display fix for uint64 bypass
-- Fixes: health bar fill, consistent text display, no flickering

-- ============================================
-- CONFIGURATION
-- ============================================
local ADDON_NAME = "BigHealthDisplay"
local VERSION = "12.1"

-- Magic marker BASES from server (must match server code!)
local MARKER_BILLION_BASE = 2000000000
local MARKER_TRILLION_BASE = 2100000000  
local MARKER_QUADRILLION_BASE = 2200000000

-- ============================================
-- CORE FUNCTIONS
-- ============================================

local function FormatLargeNumber(value)
    if not value or value == 0 then return "0" end
    
    local absValue = math.abs(value)
    local sign = value < 0 and "-" or ""
    
    if absValue >= 1000000000000000 then
        return sign .. string.format("%.2fQ", absValue / 1000000000000000)
    elseif absValue >= 1000000000000 then
        return sign .. string.format("%.2fT", absValue / 1000000000000)
    elseif absValue >= 1000000000 then
        return sign .. string.format("%.2fB", absValue / 1000000000)
    elseif absValue >= 1000000 then
        return sign .. string.format("%.1fM", absValue / 1000000)
    elseif absValue >= 1000 then
        return sign .. string.format("%.1fK", absValue / 1000)
    else
        return sign .. tostring(math.floor(absValue))
    end
end

local function CleanFormat(str)
    str = str:gsub("%.00([QKMBT])", "%1")
    str = str:gsub("%.0([QKMBT])", "%1")
    return str
end

local function FormatHealth(value)
    return CleanFormat(FormatLargeNumber(value))
end

-- Check if a maxHealth value is encoded (has marker)
local function IsEncodedHealth(displayMaxHealth)
    return displayMaxHealth and displayMaxHealth >= MARKER_BILLION_BASE
end

-- Decode the real health from server's encoded values
-- Returns: realHealth, realMaxHealth, scaleName, percentage, scaledHealth, scaledMax
local function DecodeHealth(unit)
    if not unit or not UnitExists(unit) then
        return 0, 0, nil, 0, 0, 0
    end
    
    local displayHealth = UnitHealth(unit)
    local displayMaxHealth = UnitHealthMax(unit)
    
    -- Handle 0 max health to avoid division by zero
    if not displayMaxHealth or displayMaxHealth == 0 then
        return displayHealth or 0, 0, nil, 0, displayHealth or 0, 0
    end
    
    local realHealth, realMaxHealth, scaleName, scaledHealth, scaledMax, multiplier
    
    if displayMaxHealth >= MARKER_QUADRILLION_BASE then
        scaledMax = displayMaxHealth - MARKER_QUADRILLION_BASE
        scaledHealth = displayHealth
        multiplier = 1000000000000
        scaleName = "Q"
    elseif displayMaxHealth >= MARKER_TRILLION_BASE then
        scaledMax = displayMaxHealth - MARKER_TRILLION_BASE
        scaledHealth = displayHealth
        multiplier = 1000000000
        scaleName = "T"
    elseif displayMaxHealth >= MARKER_BILLION_BASE then
        scaledMax = displayMaxHealth - MARKER_BILLION_BASE
        scaledHealth = displayHealth
        multiplier = 1000000
        scaleName = "B"
    else
        -- Normal health - no encoding
        return displayHealth, displayMaxHealth, nil, 
               math.floor((displayHealth / displayMaxHealth) * 100 + 0.5),
               displayHealth, displayMaxHealth
    end
    
    -- Calculate real values
    realHealth = scaledHealth * multiplier
    realMaxHealth = scaledMax * multiplier
    
    -- FIX: Handle "1HP" sync glitch - when scaledHealth is 1 but max is large
    -- This is almost always a sync delay, not the real health value
    if scaledHealth == 1 and scaledMax > 100 and not UnitIsDeadOrGhost(unit) then
        scaledHealth = scaledMax
        realHealth = realMaxHealth
    end
    
    -- SANITY CHECK: Clamp health to never exceed max
    -- This prevents display bugs from encoding edge cases or healing
    if scaledHealth > scaledMax then
        scaledHealth = scaledMax
        realHealth = realMaxHealth
    end
    if scaledHealth < 0 then
        scaledHealth = 0
        realHealth = 0
    end
    
    -- Calculate percentage from SCALED values
    local percentage
    if scaledMax and scaledMax > 0 then
        percentage = math.floor((scaledHealth / scaledMax) * 100 + 0.5)
    else
        percentage = 0
    end
    
    -- Clamp percentage
    if percentage > 100 then percentage = 100 end
    if percentage < 0 then percentage = 0 end
    if scaledHealth > 0 and percentage == 0 then percentage = 1 end
    
    return realHealth, realMaxHealth, scaleName, percentage, scaledHealth, scaledMax
end

-- Format for display
local function GetHealthDisplay(unit)
    local realHealth, realMaxHealth, scaleName, percentage = DecodeHealth(unit)
    return string.format("%s / %s (%d%%)", FormatHealth(realHealth), FormatHealth(realMaxHealth), percentage)
end

local function GetHealthDisplayCompact(unit)
    local realHealth, _, _, percentage = DecodeHealth(unit)
    return string.format("%s %d%%", FormatHealth(realHealth), percentage)
end

-- ============================================
-- UNIT FRAME OVERLAY SYSTEM (Target, Focus, Player, Pet)
-- Uses the SAME approach as nameplates - read from bar directly!
-- ============================================

local unitFrameOverlays = {}
local unitFrameBarsFixed = {}
local currentTargetNameplateBar = nil
local targetNameplateCache = { val = 0, max = 0, time = 0 }
local UpdateTargetNameplateBar

-- Fix a unit frame health bar to store original functions
local function FixUnitFrameBar(bar, unit)
    if not bar then return end
    
    -- If already processed, just ensure overlay is set up
    if unitFrameBarsFixed[bar] then
        return
    end
    unitFrameBarsFixed[bar] = true
    
    -- Store ORIGINAL C functions only if not already stored
    if not bar._bhOrigGetMinMax then
        bar._bhOrigSetMinMax = bar.SetMinMaxValues
        bar._bhOrigSetValue = bar.SetValue
        bar._bhOrigGetMinMax = bar.GetMinMaxValues
        bar._bhOrigGetValue = bar.GetValue
    end
    bar._bhUnit = unit
    
    -- Clean up any existing overlay first
    if bar.BigHealthOverlay then
        bar.BigHealthOverlay:Hide()
        bar.BigHealthOverlay:SetParent(nil)
        bar.BigHealthOverlay = nil
    end
    
    -- Create fresh overlay (parented to bar)
    local overlay = CreateFrame("StatusBar", nil, bar)
    overlay:SetAllPoints(bar)
    overlay:SetFrameLevel(bar:GetFrameLevel() + 2)  -- Higher level to ensure on top
    
    -- Background to cover original bar completely
    local bg = overlay:CreateTexture(nil, "BACKGROUND")
    bg:SetAllPoints(overlay)
    bg:SetTexture("Interface\\Buttons\\WHITE8X8")
    bg:SetVertexColor(0, 0, 0, 1)
    
    -- Copy texture and fill behavior from the original bar
    local texture = bar:GetStatusBarTexture()
    if texture and texture.GetTexture then
        local texturePath = texture:GetTexture()
        if texturePath then
            overlay:SetStatusBarTexture(texturePath)
        else
            overlay:SetStatusBarTexture("Interface\\TargetingFrame\\UI-StatusBar")
        end
    else
        overlay:SetStatusBarTexture("Interface\\TargetingFrame\\UI-StatusBar")
    end
    if bar.GetOrientation then
        overlay:SetOrientation(bar:GetOrientation())
    end
    if bar.GetReverseFill then
        overlay:SetReverseFill(bar:GetReverseFill())
    end
    overlay:SetStatusBarColor(0, 1, 0)
    overlay:SetMinMaxValues(0, 1000)
    overlay:SetValue(1000)
    overlay:Hide()
    
    -- Text on overlay at highest level
    local text = overlay:CreateFontString(nil, "OVERLAY", "TextStatusBarText")
    text:SetPoint("CENTER", overlay, "CENTER", 0, 0)
    text:SetDrawLayer("OVERLAY", 7)
    overlay.text = text
    
    bar.BigHealthOverlay = overlay
    unitFrameOverlays[bar] = overlay
end

-- Cache for bar values to detect glitches
local barCache = {}

-- Called when target changes
local function OnTargetChanged()
    -- Nothing needed now that cache is gone
end

-- Update unit frame overlay
local function UpdateUnitFrameOverlayFromBar(bar)
    if not bar or not bar._bhOrigGetMinMax or not bar._bhOrigGetValue then return end
    if not bar:IsVisible() then
        if bar.BigHealthOverlay then bar.BigHealthOverlay:Hide() end
        return
    end
    
    local overlay = bar.BigHealthOverlay
    if not overlay then return end
    
    local useUnitAPI = false
    local rawVal = 0
    local rawMax = 0
    local rawMin = 0
    
    local unit = bar._bhUnit
    local forceOverlay = (unit == "target" or unit == "focus" or unit == "targettarget")
    local rawFromNameplate = false

    -- For target, prefer nameplate health (matches on-screen bar)
    if unit == "target" then
        if currentTargetNameplateBar and currentTargetNameplateBar:IsVisible() then
            rawMin, rawMax = currentTargetNameplateBar:GetMinMaxValues()
            rawVal = currentTargetNameplateBar:GetValue()
            rawFromNameplate = true
            targetNameplateCache.val = rawVal or 0
            targetNameplateCache.max = rawMax or 0
            targetNameplateCache.time = GetTime()
        elseif targetNameplateCache.time and (GetTime() - targetNameplateCache.time) < 1.0 then
            rawMin = 0
            rawMax = targetNameplateCache.max
            rawVal = targetNameplateCache.val
            rawFromNameplate = true
        end
    end

    if not rawFromNameplate then
        -- Use Direct Unit API for player/pet to avoid bar lag; bar for other frames
        if unit and UnitExists(unit) and not forceOverlay then
            useUnitAPI = true
            rawVal = UnitHealth(unit)
            rawMax = UnitHealthMax(unit)
            rawMin = 0
        else
            -- Read the bar directly (authoritative for target/focus/ToT frames)
            rawMin, rawMax = bar._bhOrigGetMinMax(bar)
            rawVal = bar._bhOrigGetValue(bar)
        end
    end
    
    local now = GetTime()
    local graceWindow = 0.5

    if not rawMax or rawMax == 0 then
        -- Brief grace period to avoid showing wrong numbers during sync/retarget
        if bar._bhLastEncodedTime and (now - bar._bhLastEncodedTime) < graceWindow then
            if bar._bhLastScaledMax and bar._bhLastScaledVal and bar._bhLastMultiplier then
                overlay:SetMinMaxValues(0, bar._bhLastScaledMax > 0 and bar._bhLastScaledMax or 1)
                overlay:SetValue(bar._bhLastScaledVal)
                overlay:Show()
                if overlay.text then
                    local realHealth = bar._bhLastScaledVal * bar._bhLastMultiplier
                    local realMaxHealth = bar._bhLastScaledMax * bar._bhLastMultiplier
                    local pct = 0
                    if bar._bhLastScaledMax > 0 then
                        pct = math.floor((bar._bhLastScaledVal / bar._bhLastScaledMax) * 100 + 0.5)
                    end
                    if pct > 100 then pct = 100 end
                    if pct < 0 then pct = 0 end
                    if bar._bhLastScaledVal > 0 and pct == 0 then pct = 1 end
                    overlay.text:SetText(string.format("%s / %s (%d%%)",
                        FormatHealth(realHealth), FormatHealth(realMaxHealth), pct))
                end
                if bar.TextString then bar.TextString:SetAlpha(0) end
                local barName = bar:GetName()
                if barName then
                    local txt = _G[barName .. "Text"]
                    if txt then txt:SetAlpha(0) end
                    local txtL = _G[barName .. "TextLeft"]
                    if txtL then txtL:SetAlpha(0) end
                    local txtR = _G[barName .. "TextRight"]
                    if txtR then txtR:SetAlpha(0) end
                end
                return
            end
        end

        if forceOverlay then
            if bar.TextString then bar.TextString:SetAlpha(0) end
            local barName = bar:GetName()
            if barName then
                local txt = _G[barName .. "Text"]
                if txt then txt:SetAlpha(0) end
                local txtL = _G[barName .. "TextLeft"]
                if txtL then txtL:SetAlpha(0) end
                local txtR = _G[barName .. "TextRight"]
                if txtR then txtR:SetAlpha(0) end
            end
        end

        overlay:Hide()
        return
    end
    
    -- Check if encoded
    if rawMax >= MARKER_BILLION_BASE then
        local scaledMax, multiplier, scaleName
        
        if rawMax >= MARKER_QUADRILLION_BASE then
            scaledMax = rawMax - MARKER_QUADRILLION_BASE
            multiplier = 1000000000000
            scaleName = "Q"
        elseif rawMax >= MARKER_TRILLION_BASE then
            scaledMax = rawMax - MARKER_TRILLION_BASE
            multiplier = 1000000000
            scaleName = "T"
        else
            scaledMax = rawMax - MARKER_BILLION_BASE
            multiplier = 1000000
            scaleName = "B"
        end
        
        -- Calculate the actual percentage from the SERVER's perspective
        -- The server sends: scaledHealth / scaledMax which represents the TRUE health ratio
        -- We should trust these values directly as they ARE the authoritative source
        local useRawVal = rawVal
        local unit = bar._bhUnit
        
        -- SANITY CHECK: Ensure values are within valid bounds
        -- This handles edge cases where encoding/sync might produce invalid values
        if useRawVal < 0 then useRawVal = 0 end
        if useRawVal > scaledMax then useRawVal = scaledMax end
        
        -- FIX: Handle the "1HP" sync glitch for ALL units (player, pet, target, focus)
        -- When rawVal is 1 but scaledMax is large, this is almost always a sync delay glitch
        -- The server hasn't sent the real value yet, so we show 100% until it does
        local isPlayerOrPet = (unit == "player" or unit == "pet")
        
        -- For player/pet: Apply the 1HP fix but don't track GUID history
        if isPlayerOrPet then
            -- If we see "1" and max is large, and unit is alive, it's a sync glitch
            if useRawVal == 1 and scaledMax > 100 and not UnitIsDeadOrGhost(unit) then
                useRawVal = scaledMax
            end
        else
            -- For target/focus: Track GUID to detect target changes
            local currentGUID = UnitGUID(unit)
            
            if bar._bhLastGUID ~= currentGUID then
                bar._bhLastGUID = currentGUID
                bar._bhLastValid = nil
                bar._bhFirstUpdate = true
            end
            
            -- Apply the "1HP fix" - if value is 1 and max is large, it's a glitch
            if useRawVal == 1 and scaledMax > 100 and not UnitIsDeadOrGhost(unit) then
                if bar._bhFirstUpdate then
                    -- First frame sync delay - show 100% until we get real data
                    useRawVal = scaledMax
                elseif bar._bhLastValid then
                    -- We have a previous valid value, use it instead of glitchy 1
                    useRawVal = bar._bhLastValid
                else
                    -- No history, assume 100%
                    useRawVal = scaledMax
                end
            else
                bar._bhFirstUpdate = false
            end
            
            -- Track valid values for fallback
            if useRawVal > 1 then
                bar._bhLastValid = useRawVal
            end
        end
        
        -- Calculate percentage from the final values
        local percentage = 0
        if scaledMax > 0 then
            percentage = math.floor((useRawVal / scaledMax) * 100 + 0.5)
        end
        
        -- Final percentage sanity clamp
        if percentage > 100 then percentage = 100 end
        if percentage < 0 then percentage = 0 end
        if useRawVal > 0 and percentage == 0 then percentage = 1 end
        
        -- Copy color from original bar
        local r, g, b = bar:GetStatusBarColor()
        if r then
            overlay:SetStatusBarColor(r, g, b)
        end
        
        -- HIDE original bar text
        if bar.TextString then bar.TextString:SetAlpha(0) end
        local barName = bar:GetName()
        if barName then
            local txt = _G[barName .. "Text"]
            if txt then txt:SetAlpha(0) end
            local txtL = _G[barName .. "TextLeft"]
            if txtL then txtL:SetAlpha(0) end
            local txtR = _G[barName .. "TextRight"]
            if txtR then txtR:SetAlpha(0) end
        end
        
        -- Determine display values
        overlay:SetMinMaxValues(0, scaledMax > 0 and scaledMax or 1)
        overlay:SetValue(useRawVal)
        overlay:Show()

        -- Cache last good encoded values to prevent flicker
        bar._bhLastEncodedTime = now
        bar._bhLastScaledMax = scaledMax
        bar._bhLastScaledVal = useRawVal
        bar._bhLastMultiplier = multiplier
        
        -- Update text
        if overlay.text then
            local realHealth = useRawVal * multiplier
            local realMaxHealth = scaledMax * multiplier
            local newText = string.format("%s / %s (%d%%)", 
                FormatHealth(realHealth), FormatHealth(realMaxHealth), percentage)
            
            overlay.text:SetText(newText)
        end
    else
        -- Normal health
        if forceOverlay then
            local pct = 0
            if rawMax > 0 then
                pct = math.floor((rawVal / rawMax) * 100 + 0.5)
            end
            if pct > 100 then pct = 100 end
            if pct < 0 then pct = 0 end
            if rawVal > 0 and pct == 0 then pct = 1 end

            -- Copy color from original bar
            local r, g, b = bar:GetStatusBarColor()
            if r then
                overlay:SetStatusBarColor(r, g, b)
            end

            overlay:SetMinMaxValues(0, rawMax > 0 and rawMax or 1)
            overlay:SetValue(rawVal)
            overlay:Show()

            if overlay.text then
                overlay.text:SetText(string.format("%s / %s (%d%%)",
                    FormatHealth(rawVal), FormatHealth(rawMax), pct))
            end

            -- Always hide original bar text for target/focus/ToT
            if bar.TextString then bar.TextString:SetAlpha(0) end
            local barName = bar:GetName()
            if barName then
                local txt = _G[barName .. "Text"]
                if txt then txt:SetAlpha(0) end
                local txtL = _G[barName .. "TextLeft"]
                if txtL then txtL:SetAlpha(0) end
                local txtR = _G[barName .. "TextRight"]
                if txtR then txtR:SetAlpha(0) end
            end
            return
        end

        -- Normal health - hide overlay and restore original text
        -- If we just had encoded values, keep overlay briefly to prevent blink
        if bar._bhLastEncodedTime and (now - bar._bhLastEncodedTime) < graceWindow then
            if bar._bhLastScaledMax and bar._bhLastScaledVal and bar._bhLastMultiplier then
                overlay:SetMinMaxValues(0, bar._bhLastScaledMax > 0 and bar._bhLastScaledMax or 1)
                overlay:SetValue(bar._bhLastScaledVal)
                overlay:Show()
                if overlay.text then
                    local realHealth = bar._bhLastScaledVal * bar._bhLastMultiplier
                    local realMaxHealth = bar._bhLastScaledMax * bar._bhLastMultiplier
                    local pct = 0
                    if bar._bhLastScaledMax > 0 then
                        pct = math.floor((bar._bhLastScaledVal / bar._bhLastScaledMax) * 100 + 0.5)
                    end
                    if pct > 100 then pct = 100 end
                    if pct < 0 then pct = 0 end
                    if bar._bhLastScaledVal > 0 and pct == 0 then pct = 1 end
                    overlay.text:SetText(string.format("%s / %s (%d%%)",
                        FormatHealth(realHealth), FormatHealth(realMaxHealth), pct))
                end
                if bar.TextString then bar.TextString:SetAlpha(0) end
                local barName = bar:GetName()
                if barName then
                    local txt = _G[barName .. "Text"]
                    if txt then txt:SetAlpha(0) end
                    local txtL = _G[barName .. "TextLeft"]
                    if txtL then txtL:SetAlpha(0) end
                    local txtR = _G[barName .. "TextRight"]
                    if txtR then txtR:SetAlpha(0) end
                end
                return
            end
        end

        overlay:Hide()
        bar._bhLastGUID = nil
        bar._bhLastGoodVal = nil
        
        -- Restore original bar's text visibility
        if bar.TextString then
            bar.TextString:SetAlpha(1)
        end
        local barName = bar:GetName()
        if barName then
            local txt = _G[barName .. "Text"]
            if txt then txt:SetAlpha(1) end
            local txtL = _G[barName .. "TextLeft"]
            if txtL then txtL:SetAlpha(1) end
            local txtR = _G[barName .. "TextRight"]
            if txtR then txtR:SetAlpha(1) end
        end
    end
end

-- ============================================
-- TEXT UPDATES (simplified - overlay handles display)
-- ============================================

local function UpdateTargetHealth()
    if TargetFrameHealthBar then
        UpdateUnitFrameOverlayFromBar(TargetFrameHealthBar)
    end
end

local function UpdateFocusHealth()
    if FocusFrameHealthBar then
        UpdateUnitFrameOverlayFromBar(FocusFrameHealthBar)
    end
end

local function UpdateTargetTargetHealth()
    if TargetFrameToTHealthBar then
        UpdateUnitFrameOverlayFromBar(TargetFrameToTHealthBar)
    end
end

local function UpdatePlayerHealth()
    if PlayerFrameHealthBar then
        UpdateUnitFrameOverlayFromBar(PlayerFrameHealthBar)
    end
end

local function UpdatePetHealth()
    if PetFrameHealthBar then
        UpdateUnitFrameOverlayFromBar(PetFrameHealthBar)
    end
end

local function UpdateAllHealth()
    UpdatePlayerHealth()
    UpdateTargetHealth()
    UpdateFocusHealth()
    UpdateTargetTargetHealth()
    UpdatePetHealth()
end

-- ============================================
-- EVENT HANDLING
-- ============================================

local frame = CreateFrame("Frame", "BigHealthDisplayFrame", UIParent)
frame:RegisterEvent("UNIT_HEALTH")
frame:RegisterEvent("UNIT_MAXHEALTH")
frame:RegisterEvent("PLAYER_TARGET_CHANGED")
frame:RegisterEvent("PLAYER_FOCUS_CHANGED")
frame:RegisterEvent("PLAYER_ENTERING_WORLD")
frame:RegisterEvent("UNIT_PET")
-- Add regens to catch awkward state updates
frame:RegisterEvent("PLAYER_REGEN_DISABLED")
frame:RegisterEvent("PLAYER_REGEN_ENABLED")

frame:SetScript("OnEvent", function(self, event, unit)
    if event == "PLAYER_ENTERING_WORLD" then
        -- Fix all unit frame bars (store original functions)
        if TargetFrameHealthBar then FixUnitFrameBar(TargetFrameHealthBar, "target") end
        if FocusFrameHealthBar then FixUnitFrameBar(FocusFrameHealthBar, "focus") end
        if TargetFrameToTHealthBar then FixUnitFrameBar(TargetFrameToTHealthBar, "targettarget") end
        if PlayerFrameHealthBar then FixUnitFrameBar(PlayerFrameHealthBar, "player") end
        if PetFrameHealthBar then FixUnitFrameBar(PetFrameHealthBar, "pet") end
        
        -- Small delay using OnUpdate
        self.initDelay = 0.2
        return
    end
    
    if event == "PLAYER_TARGET_CHANGED" then
        OnTargetChanged()
        UpdateTargetNameplateBar()
        UpdateTargetHealth()
        UpdateTargetTargetHealth()
        return
    end
    
    if event == "PLAYER_FOCUS_CHANGED" then
        UpdateFocusHealth()
        return
    end
    
    if event == "UNIT_PET" then
        if PetFrameHealthBar then FixUnitFrameBar(PetFrameHealthBar, "pet") end
        UpdatePetHealth()
        return
    end

    if event == "PLAYER_REGEN_DISABLED" or event == "PLAYER_REGEN_ENABLED" then
        UpdateAllHealth()
        return
    end
    
    -- UNIT_HEALTH or UNIT_MAXHEALTH
    if unit == "target" then
        UpdateTargetHealth()
    elseif unit == "focus" then
        UpdateFocusHealth()
    elseif unit == "targettarget" then
        UpdateTargetTargetHealth()
    elseif unit == "player" then
        UpdatePlayerHealth()
    elseif unit == "pet" then
        UpdatePetHealth()
    end
end)

-- Handle delayed initialization
frame:SetScript("OnUpdate", function(self, elapsed)
    if self.initDelay then
        self.initDelay = self.initDelay - elapsed
        if self.initDelay <= 0 then
            self.initDelay = nil
            UpdateAllHealth()
        end
    end
end)

-- ============================================
-- CONTINUOUS UPDATE - Read from bars directly like nameplates
-- ============================================

local unitUpdateFrame = CreateFrame("Frame")
unitUpdateFrame:SetScript("OnUpdate", function(self, delta)
    -- Update unit frame overlays every frame by reading from bars directly
    if TargetFrameHealthBar then
        UpdateUnitFrameOverlayFromBar(TargetFrameHealthBar)
    end
    if FocusFrameHealthBar then
        UpdateUnitFrameOverlayFromBar(FocusFrameHealthBar)
    end
    if TargetFrameToTHealthBar then
        UpdateUnitFrameOverlayFromBar(TargetFrameToTHealthBar)
    end
    if PlayerFrameHealthBar then
        UpdateUnitFrameOverlayFromBar(PlayerFrameHealthBar)
    end
    if PetFrameHealthBar then
        UpdateUnitFrameOverlayFromBar(PetFrameHealthBar)
    end
end)

-- ============================================
-- TOOLTIP SUPPORT
-- ============================================

local function UpdateTooltipStatusBar(unit)
    local bar = GameTooltipStatusBar
    if not bar or not unit or not UnitExists(unit) then return end

    local displayHealth = UnitHealth(unit)
    local displayMax = UnitHealthMax(unit)
    if not displayMax or displayMax == 0 then return end

    if IsEncodedHealth(displayMax) then
        local _, _, _, _, scaledHealth, scaledMax = DecodeHealth(unit)
        bar:SetMinMaxValues(0, scaledMax > 0 and scaledMax or 1)
        bar:SetValue(scaledHealth or 0)
    else
        bar:SetMinMaxValues(0, displayMax)
        bar:SetValue(displayHealth or 0)
    end
    bar:Show()
end

local tooltipFrame = CreateFrame("Frame")
tooltipFrame:SetScript("OnUpdate", function(self, delta)
    self.timer = (self.timer or 0) + delta
    if self.timer < 0.1 then return end
    self.timer = 0
    
    if not UnitExists("mouseover") then return end
    UpdateTooltipStatusBar("mouseover")
    if not IsEncodedHealth(UnitHealthMax("mouseover")) then return end
    
    local healthText = GetHealthDisplay("mouseover")
    
    for i = 2, GameTooltip:NumLines() do
        local line = _G["GameTooltipTextLeft" .. i]
        if line then
            local text = line:GetText()
            if text and text:match("^%d") then
                line:SetText(healthText)
                GameTooltip:Show()
                break
            end
        end
    end
end)

-- ============================================
-- NAMEPLATE SUPPORT (3.3.5 / WotLK) - REWRITTEN
-- ============================================

-- In 3.3.5, nameplates are structured as:
-- WorldFrame
--   └─ Nameplate (unnamed Frame)
--       ├─ Border/artwork regions
--       ├─ Name FontString
--       └─ Bars Frame (child frame containing bars)
--           ├─ Health Bar (StatusBar) - usually first
--           └─ Cast Bar (StatusBar)

local fixedBars = {}

-- Recursively find all StatusBars in a frame
local function FindAllStatusBars(frame, results)
    results = results or {}
    
    if not frame then return results end
    
    -- Check if this frame is a StatusBar
    if frame.GetObjectType and frame:GetObjectType() == "StatusBar" then
        table.insert(results, frame)
    end
    
    -- Check children
    if frame.GetNumChildren then
        for i = 1, frame:GetNumChildren() do
            local child = select(i, frame:GetChildren())
            if child then
                FindAllStatusBars(child, results)
            end
        end
    end
    
    return results
end

-- Check if a frame is likely a nameplate
local function IsNameplate(frame)
    if not frame then return false end
    if frame:GetName() then return false end  -- Nameplates are unnamed
    
    -- Must have regions (name text, borders, etc)
    local numRegions = frame:GetNumRegions()
    if numRegions < 1 then return false end
    
    -- Check for a name FontString
    local hasName = false
    for i = 1, numRegions do
        local region = select(i, frame:GetRegions())
        if region and region:GetObjectType() == "FontString" then
            local text = region:GetText()
            if text and text ~= "" then
                hasName = true
                break
            end
        end
    end
    
    if not hasName then return false end
    
    -- Must have StatusBars (health bar)
    local statusBars = FindAllStatusBars(frame)
    if #statusBars < 1 then return false end
    
    return true
end

local function GetNameplateName(frame)
    if not frame or not frame.GetNumRegions then return nil end
    local numRegions = frame:GetNumRegions()
    for i = 1, numRegions do
        local region = select(i, frame:GetRegions())
        if region and region:GetObjectType() == "FontString" then
            local text = region:GetText()
            if text and text ~= "" then
                return text
            end
        end
    end
    return nil
end

local function IsTargetNameplate(frame)
    if not frame or not frame.GetNumRegions then return false end
    local numRegions = frame:GetNumRegions()
    for i = 1, numRegions do
        local region = select(i, frame:GetRegions())
        if region and region.GetTexture and region:IsShown() then
            local tex = region:GetTexture()
            if type(tex) == "string" then
                if tex:find("TargetingFrame") or (tex:find("Nameplate") and tex:find("Target")) then
                    return true
                end
            end
        end
    end
    return false
end

UpdateTargetNameplateBar = function()
    currentTargetNameplateBar = nil
    local targetName = UnitName("target")
    if not targetName then return end

    local worldFrame = WorldFrame
    if not worldFrame then return end

    local fallbackBar = nil
    local matchCount = 0

    for i = 1, select("#", worldFrame:GetChildren()) do
        local child = select(i, worldFrame:GetChildren())
        if child and IsNameplate(child) then
            local nameText = GetNameplateName(child)
            if nameText == targetName then
                local bars = FindAllStatusBars(child)
                if #bars > 0 then
                    matchCount = matchCount + 1
                    if not fallbackBar then fallbackBar = bars[1] end
                    if IsTargetNameplate(child) then
                        currentTargetNameplateBar = bars[1]
                        return
                    end
                end
            end
        end
    end

    if matchCount == 1 then
        currentTargetNameplateBar = fallbackBar
    end
end

-- Fix a StatusBar that's a health bar - OVERLAY APPROACH (no flicker)
local function FixStatusBar(bar)
    if not bar or fixedBars[bar] then return end
    
    fixedBars[bar] = true
    
    -- Store ORIGINAL C functions
    local origSetMinMax = bar.SetMinMaxValues
    local origSetValue = bar.SetValue
    local origGetMinMaxValues = bar.GetMinMaxValues
    local origGetValue = bar.GetValue
    
    bar._bhOrigSetMinMax = origSetMinMax
    bar._bhOrigSetValue = origSetValue
    bar._bhOrigGetMinMax = origGetMinMaxValues
    bar._bhOrigGetValue = origGetValue
    
    -- State tracking
    bar._bhEncodedMax = nil
    bar._bhScaledMax = nil
    bar._bhScaledHealth = nil
    bar._bhMultiplier = nil
    bar._bhScale = nil
    
    -- CREATE OUR OWN OVERLAY BAR that we control completely
    if not bar.BigHealthOverlay then
        -- Create overlay statusbar
        local overlay = CreateFrame("StatusBar", nil, bar)
        overlay:SetAllPoints(bar)
        overlay:SetFrameLevel(bar:GetFrameLevel() + 1)
        
        -- Copy the texture from original bar if possible
        local texture = bar:GetStatusBarTexture()
        if texture then
            local texturePath = texture:GetTexture()
            if texturePath then
                overlay:SetStatusBarTexture(texturePath)
            else
                overlay:SetStatusBarTexture("Interface\\TargetingFrame\\UI-StatusBar")
            end
        else
            overlay:SetStatusBarTexture("Interface\\TargetingFrame\\UI-StatusBar")
        end
        
        overlay:SetStatusBarColor(0, 1, 0)  -- Green by default
        overlay:SetMinMaxValues(0, 100)
        overlay:SetValue(100)
        overlay:Hide()  -- Hidden until we have encoded health
        
        bar.BigHealthOverlay = overlay
    end
    
    -- Create text overlay
    if not bar.BigHealthText then
        bar.BigHealthText = bar.BigHealthOverlay:CreateFontString(nil, "OVERLAY")
        bar.BigHealthText:SetFont(STANDARD_TEXT_FONT, 7, "OUTLINE")
        bar.BigHealthText:SetPoint("CENTER", bar.BigHealthOverlay, "CENTER", 0, 0)
        bar.BigHealthText:SetTextColor(1, 1, 1, 1)
    end
    
    -- Helper to decode max value
    local function DecodeMaxValue(maxVal)
        if maxVal and maxVal >= MARKER_BILLION_BASE then
            local scaledMax, multiplier, scale
            if maxVal >= MARKER_QUADRILLION_BASE then
                scaledMax = maxVal - MARKER_QUADRILLION_BASE
                multiplier = 1000000000000
                scale = "Q"
            elseif maxVal >= MARKER_TRILLION_BASE then
                scaledMax = maxVal - MARKER_TRILLION_BASE
                multiplier = 1000000000
                scale = "T"
            else
                scaledMax = maxVal - MARKER_BILLION_BASE
                multiplier = 1000000
                scale = "B"
            end
            return true, scaledMax, multiplier, scale
        end
        return false, maxVal, 1, nil
    end
    
    -- Function to update our overlay
    local function UpdateOverlay(health, scaledMax, multiplier, encodedMax)
        local overlay = bar.BigHealthOverlay
        if not overlay then return end
        
        -- Copy color from original bar
        local r, g, b = bar:GetStatusBarColor()
        if r then
            overlay:SetStatusBarColor(r, g, b)
        end
        
        -- Set our overlay values
        overlay:SetMinMaxValues(0, scaledMax > 0 and scaledMax or 1)
        overlay:SetValue(health)
        overlay:Show()
        
        -- Hide original bar's fill by making it tiny
        -- (Can't actually hide it, but we cover it with overlay)
        
        -- Update text
        if bar.BigHealthText then
            local pct = 0
            if scaledMax > 0 then
                pct = math.floor((health / scaledMax) * 100 + 0.5)
            end
            if pct > 100 then pct = 100 end
            if pct < 0 then pct = 0 end
            local realHealth = health * multiplier
            bar.BigHealthText:SetText(FormatHealth(realHealth) .. " " .. pct .. "%")
            bar.BigHealthText:Show()
        end
    end
    
    -- Function to hide overlay and show normal bar
    local function HideOverlay()
        if bar.BigHealthOverlay then
            bar.BigHealthOverlay:Hide()
        end
        if bar.BigHealthText then
            bar.BigHealthText:Hide()
        end
    end
    
    -- Store update function for refresh loop
    bar._bhUpdateOverlay = UpdateOverlay
    bar._bhHideOverlay = HideOverlay
    bar._bhDecodeMax = DecodeMaxValue
end

-- Process a nameplate
local function ProcessNameplate(nameplate)
    if not nameplate or nameplate._bhProcessed then return end
    
    if not IsNameplate(nameplate) then return end
    
    nameplate._bhProcessed = true
    
    -- Find and fix ALL StatusBars (not just colored ones)
    local statusBars = FindAllStatusBars(nameplate)
    
    for _, bar in ipairs(statusBars) do
        FixStatusBar(bar)
    end
end

-- Global scan for nameplates
local function ScanAllNameplates()
    local worldFrame = WorldFrame
    if not worldFrame then return end
    
    for i = 1, select("#", worldFrame:GetChildren()) do
        local child = select(i, worldFrame:GetChildren())
        if child then
            ProcessNameplate(child)
        end
    end

    UpdateTargetNameplateBar()
end

-- Main update loop - reads raw values and updates our overlay
local function RefreshAllNameplateBars()
    for bar, _ in pairs(fixedBars) do
        if bar and bar:IsVisible() and bar._bhOrigGetMinMax and bar._bhOrigGetValue then
            -- Get RAW values from the bar (what the game set)
            local rawMin, rawMax = bar._bhOrigGetMinMax(bar)
            local rawVal = bar._bhOrigGetValue(bar)
            
            -- Check if this is encoded health
            if rawMax and rawMax >= MARKER_BILLION_BASE then
                -- Decode it
                local isEncoded, scaledMax, multiplier, scale = bar._bhDecodeMax(rawMax)
                
                if isEncoded and bar._bhUpdateOverlay then
                    -- Update our overlay with correct values
                    bar._bhEncodedMax = rawMax
                    bar._bhScaledMax = scaledMax
                    bar._bhMultiplier = multiplier
                    bar._bhScale = scale
                    bar._bhScaledHealth = rawVal
                    
                    bar._bhUpdateOverlay(rawVal, scaledMax, multiplier, rawMax)
                end
            else
                -- Normal health - hide our overlay, show original
                if bar._bhHideOverlay then
                    bar._bhHideOverlay()
                end
                bar._bhEncodedMax = nil
                bar._bhScaledMax = nil
            end
        end
    end
end

-- High-frequency update loop
local scanFrame = CreateFrame("Frame")
scanFrame.elapsed = 0
scanFrame.scanElapsed = 0
scanFrame:SetScript("OnUpdate", function(self, delta)
    self.elapsed = self.elapsed + delta
    self.scanElapsed = self.scanElapsed + delta
    
    -- Refresh overlays EVERY FRAME for zero flicker
    RefreshAllNameplateBars()
    
    -- Scan for new nameplates less frequently
    if self.scanElapsed >= 0.1 then
        self.scanElapsed = 0
        ScanAllNameplates()
    end
end)

-- ============================================
-- SLASH COMMANDS
-- ============================================

SLASH_BIGHEALTH1 = "/bighealth"
SLASH_BIGHEALTH2 = "/bh"
SlashCmdList["BIGHEALTH"] = function(msg)
    if msg == "test" then
        local testValues = {
            999, 1500, 25000, 1500000, 25000000,
            1500000000, 25000000000, 
            1500000000000, 25000000000000,
        }
        
        print("|cff00ff00=== BigHealthDisplay Format Test ===|r")
        for _, v in ipairs(testValues) do
            print(string.format("  |cffaaaaaa%s|r = |cff00ffff%s|r", tostring(v), FormatHealth(v)))
        end
    elseif msg == "info" then
        if UnitExists("target") then
            local realHealth, realMaxHealth, scaleName, percentage, scaledHealth, scaledMax = DecodeHealth("target")
            local displayHealth = UnitHealth("target")
            local displayMaxHealth = UnitHealthMax("target")
            
            print("|cff00ff00=== Target Health Info ===|r")
            print(string.format("  Raw from server: %d / %d", displayHealth, displayMaxHealth))
            print(string.format("  Scaled values: %d / %d", scaledHealth, scaledMax))
            print(string.format("  Real health: %s / %s", FormatHealth(realHealth), FormatHealth(realMaxHealth)))
            print(string.format("  Percentage: %d%%", percentage))
            print(string.format("  Scale: %s", scaleName or "Normal"))
            print(string.format("  Encoded: %s", IsEncodedHealth(displayMaxHealth) and "Yes" or "No"))
        else
            print("No target selected")
        end
    elseif msg == "refresh" then
        UpdateAllHealth()
        ScanAllNameplates()
        -- Reset custom history
        if TargetFrameHealthBar then TargetFrameHealthBar._bhLastValid = nil end
        if FocusFrameHealthBar then FocusFrameHealthBar._bhLastValid = nil end
        print("|cff00ff00BigHealthDisplay|r: Refreshed all health displays and nameplates")
    elseif msg == "np" or msg == "nameplates" then
        -- Debug nameplate info
        local count = 0
        local processed = 0
        local barsFixed = 0
        local encodedBars = 0
        local worldFrame = WorldFrame
        if worldFrame then
            for i = 1, select("#", worldFrame:GetChildren()) do
                local frame = select(i, worldFrame:GetChildren())
                if frame and frame:GetName() == nil then
                    local bars = FindAllStatusBars(frame)
                    if #bars > 0 then
                        count = count + 1
                        if frame._bhProcessed then
                            processed = processed + 1
                        end
                    end
                end
            end
        end
        for bar, _ in pairs(fixedBars) do
            barsFixed = barsFixed + 1
            if bar._bhEncodedMax then
                encodedBars = encodedBars + 1
            end
        end
        print("|cff00ff00=== Nameplate Debug ===|r")
        print(string.format("  Potential nameplates: %d", count))
        print(string.format("  Processed: %d", processed))
        print(string.format("  StatusBars hooked: %d", barsFixed))
        print(string.format("  Bars with encoded health: %d", encodedBars))
    elseif msg == "npdetail" then
        -- Detailed bar info
        print("|cff00ff00=== Nameplate Bar Details ===|r")
        local idx = 0
        for bar, _ in pairs(fixedBars) do
            idx = idx + 1
            if idx <= 5 then  -- Limit to 5 bars
                local min, max = bar:GetMinMaxValues()
                local val = bar:GetValue()
                local encoded = bar._bhEncodedMax or "none"
                local scaled = bar._bhScaledMax or "none"
                print(string.format("  Bar %d: val=%s min=%s max=%s", idx, tostring(val), tostring(min), tostring(max)))
                print(string.format("         encoded=%s scaled=%s", tostring(encoded), tostring(scaled)))
            end
        end
        if idx == 0 then
            print("  No bars hooked yet")
        end
    else
        print("|cff00ff00BigHealthDisplay|r v" .. VERSION)
        print("  |cffff9900/bh test|r - Test formatting")
        print("  |cffff9900/bh info|r - Show target health info")
        print("  |cffff9900/bh refresh|r - Force refresh all")
        print("  |cffff9900/bh np|r - Nameplate debug info")
        print("  |cffff9900/bh npdetail|r - Detailed bar values")
    end
end

-- ============================================
-- INITIALIZATION
-- ============================================

print("|cff00ff00BigHealthDisplay v" .. VERSION .. "|r loaded!")
print("  Health bars + nameplates now work with B/T/Q values")
print("  Type |cffff9900/bh|r for commands")
